const model = require("../models");
const Tales = model.tales;

const { ErrorResponse, SuccessResponse } = require("../helpers/ResponseHelper");
const { createAddress } = require("../services/addressService");
const { Op } = require("sequelize");
const address = model.address;

const CreateTales = async (req, res) => {
  try {
    if (!req.body.name) {
      return ErrorResponse(res, "Name is required", 422);
    }

    if (!req.body.video_url) {
      return ErrorResponse(res, "Video URL is required", 422);
    }

    if (!req.body.video_type) {
      return ErrorResponse(res, "Video Type is required", 422);
    }

    let address = null;

    if (req.body.address) {
      address = await createAddress(req.body.address);
    }

    let newTale = await Tales.create({
      ...req.body,
      published_at: new Date(),
      user: req.user?.role === "admin" ? null : req.user.id,
      address: address?.id,
    });

    return SuccessResponse(res, "Tale has been added", newTale);
  } catch (e) {
    return ErrorResponse(res, e.message, 500);
  }
};

const SaveTales = async (req, res) => {
  try {
    if (!req.body.name) {
      return ErrorResponse(res, "Name is required", 422);
    }

    if (!req.body.video_url) {
      return ErrorResponse(res, "Video URL is required", 422);
    }

    if (!req.body.video_type) {
      return ErrorResponse(res, "Video Type is required", 422);
    }

    let address = null;

    if (req.body.address) {
      address = await createAddress(req.body.address);
    }

    let newTale = await Tales.create({
      ...req.body,
      published_at: null,
      user: req.user?.role === "admin" ? null : req.user.id,
      address: address?.id,
    });

    return SuccessResponse(res, "Tale has been added", newTale);
  } catch (e) {
    return ErrorResponse(res, e.message, 500);
  }
};

const GetTales = async (req, res) => {
  try {
    let dataPerPage = 20;

    let userQuery = {
      where: {
        tales_id: null,
        published_at: { [Op.not]: null },
      },

      include: [
        {
          model: address,
          as: "tale_address",
        },
      ],
    };

    let addressQuery = {};

    if (req.query) {
      if (req.query.dataPerPage) {
        dataPerPage = parseInt(req.query.dataPerPage);
      }
      if (req.query.getNestedTales == "true") {
        userQuery.include.push({
          model: Tales,
          as: "children_tales",
          include: [{ model: address, as: "tale_address" }],
        });
      }

      if (req.query.is_moment == "true") {
        userQuery.where.is_moment = true;
      }

      if (req.query.country) {
        addressQuery.country = req.query.country;
      }
      if (req.query.iso_code) {
        addressQuery.iso_code = req.query.iso_code;
      }

      if (req.query.state) {
        addressQuery.state = req.query.state;
      }

      if (req.query.zip) {
        addressQuery.zip = req.query.zip;
      }
    }

    if (Object.keys(addressQuery).length > 0) {
      userQuery.include[0].where = addressQuery;
    }

    userQuery.limit = dataPerPage;
    userQuery.offset =
      ((req.query.page > 0 ? req.query.page : 1) - 1) * dataPerPage;

    let tales = await Tales.findAndCountAll({
      ...userQuery,
    });
    return SuccessResponse(res, "All tales fetched", {
      tales: tales.rows,
      currentPage: parseInt(req.query.page) || 1,
      totalPage: Math.ceil(tales.count / dataPerPage),
    });
  } catch (e) {
    return ErrorResponse(res, e.message, 500);
  }
};

const PublishTales = async (req, res) => {
  try {
    let tale_id = req.params.tale_id;

    let myTale = await Tales.findOne({
      where: {
        id: tale_id,
        published_at: null,
      },
    });

    if (!myTale) {
      return ErrorResponse(res, "Tale doesnot exist", 404);
    }

    myTale.published_at = new Date();
    await myTale.save();
    return res.json({ message: "Tale has been published" });
  } catch (e) {
    return ErrorResponse(res, e.message, 500);
  }
};

module.exports = {
  CreateTales,
  SaveTales,
  GetTales,
  PublishTales,
};
