const { ErrorResponse, SuccessResponse } = require("../helpers/ResponseHelper");
const bcrypt = require("bcrypt");
const jwt = require("jsonwebtoken");

const db = require("../models");
const {
  registerValidate,
  loginValidate,
  verifyTokenValidate,
  forgotPasswordValidate,
  verifyForgotOTPValidate,
  changePasswordValidate,
} = require("../validations/AuthValidations");
const { Op } = require("sequelize");
const User = db.Users;
const Session = db.UserSessions;

// User Registration

const RegisterUser = async (req, res) => {
  try {
    let { email, phone, fullName, password, country_code, country_iso } =
      req.body;

    let { error } = registerValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    if (!email || !phone) {
      return ErrorResponse(res, "phone or Email is required", 400);
    }

    let foundUser = "";

    foundUser = await User.findOne({
      where: { [Op.or]: { email: email, phone: phone } },
    });

    if (foundUser) {
      return ErrorResponse(res, "Email or phone in use");
    }
    foundUser = null;

    const salt = await bcrypt.genSalt(10);
    let hashedPassword = await bcrypt.hash(password, salt);

    // var mailformat = /^\w+([\.-]?\w+)*@\w+([\.-]?\w+)*(\.\w{2,3})+$/;

    let newUser = "";

    let myUsername = fullName?.split(" ")[0];

    while (true) {
      let user = await User.findOne({ where: { username: myUsername } });

      if (user) {
        myUsername = myUsername + Math.floor(Math.random() * 999 + 1);
      } else {
        break;
      }
    }

    newUser = await User.create({
      email: email,
      fullName,
      password: hashedPassword,
      username: myUsername,
      otp: 1234,
      phone: phone,
      country_code,
      country_iso,
    });

    return SuccessResponse(res, "User has been registered", newUser);
  } catch (e) {
    console.log(e);
    return ErrorResponse(res, e.message);
  }
};

const VerifyToken = async (req, res) => {
  try {
    let { loginKey, otp } = req.body;
    let { error } = verifyTokenValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    let user = await User.findOne({
      where: {
        [Op.or]: [
          { email: loginKey },
          { username: loginKey },
          { phone: loginKey },
        ],
      },
    });

    if (!user) {
      return ErrorResponse(res, "User doesnot exist", 404);
    }

    if (user.is_verified) {
      return ErrorResponse(res, "User already verified", 500);
    }

    if (user.otp !== otp) {
      return ErrorResponse(res, "OTP doesnot match", 500);
    }

    user.update({
      is_verified: true,
      otp: null,
    });

    user.save();

    return SuccessResponse(res, "Otp has been verified");
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

const LoginUser = async (req, res) => {
  try {
    let { loginKey, password, device_type, fcm_token, device_id } = req.body;

    let { error } = loginValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    let user = await User.findOne({
      where: {
        [Op.or]: [
          { email: loginKey },
          { username: loginKey },
          { phone: loginKey },
        ],
      },
    });

    if (!user) {
      return ErrorResponse(res, "User doesnot exist", 404);
    }

    if (!user.is_verified) {
      return ErrorResponse(res, "User is not verified", 500);
    }

    let isMatch = await bcrypt.compare(password, user.password);

    if (!isMatch) {
      return ErrorResponse(res, "Password doesnot match", 400);
    }

    const payload = {
      id: user.id,
      role: user.role,
    };

    const accessToken = jwt.sign(
      {
        ...payload,
      },
      process.env.SECRET_KEY,
      {
        expiresIn: "30m",
      }
    );

    const refreshToken = jwt.sign(
      {
        ...payload,
      },
      process.env.REFRESH_SECRET_KEY,
      { expiresIn: "3d" }
    );

    let userSession = await Session.findOne({
      where: {
        user_id: user?.id,
        device_id: device_id,
      },
    });

    if (!userSession) {
      await Session.create({
        user_id: user?.id,
        device_id: device_id,
        device_type: device_type,
        fcm_token: fcm_token,
        ip_address:
          req.headers["x-forwarded-for"] || req.socket.remoteAddress || null,
        access_token: accessToken,
        refresh_token: refreshToken,
      });
    }

    res.setHeader("Refresh-Token", refreshToken);
    return SuccessResponse(res, "Login Success", user, {
      accessToken: accessToken,
    });
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

// Forgot password
const ForgotPassword = async (req, res) => {
  try {
    let { emailOrPhone } = req.body;

    let { error } = forgotPasswordValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    let user = await User.findOne({
      where: {
        [Op.or]: [{ email: emailOrPhone }, { phone: emailOrPhone }],
      },
    });

    if (!user) {
      return ErrorResponse(res, "User doesnot exist", 404);
    }

    if (!user.is_verified) {
      return ErrorResponse(res, "User is not verified", 500);
    }

    await user.update({
      otp: 1234,
    });
    await user.save();
    return SuccessResponse(res, "Otp has been sent successfully");
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

//
const VerifyForgotOTP = async (req, res) => {
  try {
    let { emailOrPhone, otp } = req.body;
    let { error } = verifyForgotOTPValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    let user = await User.findOne({
      where: {
        [Op.or]: [{ email: emailOrPhone }, { phone: emailOrPhone }],
      },
    });

    if (!user) {
      return ErrorResponse(res, "User doesnot exist", 404);
    }

    if (user.otp !== otp) {
      return ErrorResponse(res, "OTP doesnot match", 500);
    }

    return SuccessResponse(res, "Otp has been verified");
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

// Change Password
const ChangePassword = async (req, res) => {
  try {
    let { error } = changePasswordValidate.validate(req.body);

    if (error) {
      return ErrorResponse(res, error.message, 400);
    }

    let { emailOrPhone, otp, password } = req.body;

    let user = await User.findOne({
      where: {
        [Op.or]: [{ email: emailOrPhone }, { phone: emailOrPhone }],
      },
    });

    if (!user) {
      return ErrorResponse(res, "User doesnot exist", 404);
    }

    if (user.otp !== otp) {
      return ErrorResponse(res, "OTP doesnot match", 500);
    }

    const salt = await bcrypt.genSalt(10);
    let hashedPassword = await bcrypt.hash(password, salt);

    await user.update({
      password: hashedPassword,
      otp: null,
    });
    await user.save();

    return SuccessResponse(res, "Password has been updated successfully");
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

// Refresh Token

const RefreshToken = (req, res) => {
  try {
    let refreshToken = req.header("Refresh-Token");

    if (refreshToken) {
      // Destructuring refreshToken from cookie

      // Verifying refresh token
      jwt.verify(
        refreshToken,
        process.env.REFRESH_SECRET_KEY,
        (err, decoded) => {
          if (err) {
            // Wrong Refesh Token
            return res.status(401).json({ message: "Unauthorized" });
          } else {
            // Correct token we send a new access token
            const accessToken = jwt.sign(
              {
                id: decoded?.user?.id,
              },
              process.env.SECRET_KEY,
              {
                expiresIn: "10m",
              }
            );
            return res.json({ accessToken });
          }
        }
      );
    } else {
      return res.status(401).json({ message: "Unauthorized" });
    }
  } catch (e) {
    return ErrorResponse(res, e.message);
  }
};

module.exports = {
  RegisterUser,
  LoginUser,
  VerifyToken,
  ForgotPassword,
  VerifyForgotOTP,
  ChangePassword,
  RefreshToken,
};
